/*
    (c) 2001-2005 Soren Roug

    This file is part of Osnine.

    Osnine is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    Osnine is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with Osnine; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/
/*
 * This program is a unit-test of negate opcode in the 6809 CPU
 */

#include <cppunit/config/SourcePrefix.h>
#include "InstructionsTest.h"
#include "typedefs.h"
#include "misc.h"
#include "mc6809.h"
#include <stdio.h>

#define LOCATION  0x1e20

// Registers the fixture into the 'registry'
CPPUNIT_TEST_SUITE_REGISTRATION( InstructionsTest );

void 
InstructionsTest::setUp()
{
    myTestCPU = new mc6809();
}


void 
InstructionsTest::tearDown()
{
}



/**
 * Helper routine to load a small program.
 */
void 
InstructionsTest::loadProg(Byte *instructions, int arraySize)
{
    Word respc;

    // Write the PC start into the top two bytes
    myTestCPU->write_word((Word)0xfffe, LOCATION);
    respc = myTestCPU->read_word((Word)0xfffe);
    CPPUNIT_ASSERT_EQUAL( LOCATION, (int)respc );

    for (int a = 0; a < arraySize; a++) {
        myTestCPU->write((Word)a + LOCATION, instructions[a]);
    }
    myTestCPU->reset();
    myTestCPU->cc.all = 0;
}
//----------------- Tests ----------------------------------------

/**
 * Test the BRA - branch always instruction.
 */
void
InstructionsTest::testBRAForward()
{
    Byte instructions[] = {
        0x20, // BRA
        17 // Jump forward 17 bytes
    };

    // Negate 0
    loadProg(instructions, sizeof instructions);
    myTestCPU->execute();
    // The size of the instruction is 2 bytes.
    CPPUNIT_ASSERT_EQUAL( LOCATION + 2 + 17, (int)myTestCPU->pc );

}

void
InstructionsTest::testBRABackward()
{
    Byte instructions[] = {
        0x20, // BRA
        170 // Jump backward 170 - 256 = 86 bytes
    };

    // Negate 0
    loadProg(instructions, sizeof instructions);
    myTestCPU->execute();
    // The size of the instruction is 2 bytes.
    CPPUNIT_ASSERT_EQUAL( LOCATION + 2 - 86, (int)myTestCPU->pc );

}

/**
 * Test the CMP - Compare - instruction.
 */
void
InstructionsTest::testCMP()
{
    // Test indirect mode: CMPA ,Y+
    // We're subtracting 0xff from 0xff and incrementing Y
    //
    // Set up a byte to test at address 0x205
    myTestCPU->write((Word)0x205, (Byte)0xff);
    // Set register Y to point to that location
    myTestCPU->y = 0x205;
    myTestCPU->a = 0xff;
    // Two bytes of instruction
    myTestCPU->write((Word)0xB00, (Byte)0xA1);
    myTestCPU->write((Word)0xB01, (Byte)0xA0);
    myTestCPU->pc = 0xB00;
    myTestCPU->execute();
    CPPUNIT_ASSERT_EQUAL( 0x206, (int)myTestCPU->y );
    CPPUNIT_ASSERT_EQUAL( 0, (int)myTestCPU->cc.bit.n );
    CPPUNIT_ASSERT_EQUAL( 1, (int)myTestCPU->cc.bit.z );
    CPPUNIT_ASSERT_EQUAL( 0, (int)myTestCPU->cc.bit.v );
    CPPUNIT_ASSERT_EQUAL( 0, (int)myTestCPU->cc.bit.c );

}

/**
 * Test the JSR - Jump to Subroutine - instruction.
 */
void
InstructionsTest::testJSR()
{
    // Test INDEXED mode:   JSR   D,Y
    //
    // Set up a word to test at address 0x205
    myTestCPU->write_word((Word)0x205, (Word)0x03ff);
    // Set register D to five
    myTestCPU->d = 0x105;
    // Set register Y to point to that location minus 5
    myTestCPU->y = 0x200;
    // Set register S to point to 0x915
    myTestCPU->s = 0x915;
    // Two bytes of instruction
    myTestCPU->write((Word)0xB00, (Byte)0xAD);
    myTestCPU->write((Word)0xB01, (Byte)0xAB);
    myTestCPU->write((Word)0xB02, (Byte)0x11); // Junk
    myTestCPU->write((Word)0xB03, (Byte)0x22); // Junk
    myTestCPU->pc = 0xB00;
    myTestCPU->execute();
    CPPUNIT_ASSERT_EQUAL( 0x200, (int)myTestCPU->y );
    CPPUNIT_ASSERT_EQUAL( 0x105, (int)myTestCPU->d );
    CPPUNIT_ASSERT_EQUAL( 0x913, (int)myTestCPU->s );
    CPPUNIT_ASSERT_EQUAL( 0x305, (int)myTestCPU->pc );

}

/**
 * Test the LDB - Load into B - instruction.
 */
void
InstructionsTest::testLDB()
{
    // Test INDEXED mode:   LDB   A,S
    //
    // Set up a word to test at address 0x205
    myTestCPU->write_word((Word)0x202, (Word)0xb3ff);
    // Set register A to the offset
    myTestCPU->a = 0x02;
    // Set register S to point to that location minus 2
    myTestCPU->s = 0x200;
    // Two bytes of instruction
    myTestCPU->write((Word)0xB00, (Byte)0xE6);
    myTestCPU->write((Word)0xB01, (Byte)0xE6);
    myTestCPU->pc = 0xB00;
    myTestCPU->execute();
    CPPUNIT_ASSERT_EQUAL( 0x200, (int)myTestCPU->s );
    CPPUNIT_ASSERT_EQUAL( 0xb3, (int)myTestCPU->b );
    CPPUNIT_ASSERT_EQUAL( 0xB02, (int)myTestCPU->pc );
    CPPUNIT_ASSERT_EQUAL( 1, (int)myTestCPU->cc.bit.n );
    CPPUNIT_ASSERT_EQUAL( 0, (int)myTestCPU->cc.bit.z );
    CPPUNIT_ASSERT_EQUAL( 0, (int)myTestCPU->cc.bit.v );
    
    // Test INDEXED mode:   LDB   A,S where A is negative
    //
    // Set up a word to test at address 0x205
    myTestCPU->write_word((Word)0x202, (Word)0x73ff);
    // Set register A to the offset
    myTestCPU->a = 0xF2;
    // Set register S to point to that location minus 2
    myTestCPU->s = 0x210;
    // Two bytes of instruction
    myTestCPU->write((Word)0xB00, (Byte)0xE6);
    myTestCPU->write((Word)0xB01, (Byte)0xE6);
    myTestCPU->pc = 0xB00;
    myTestCPU->execute();
    CPPUNIT_ASSERT_EQUAL( 0x210, (int)myTestCPU->s );
    CPPUNIT_ASSERT_EQUAL( 0x73, (int)myTestCPU->b );
    CPPUNIT_ASSERT_EQUAL( 0xB02, (int)myTestCPU->pc );
    CPPUNIT_ASSERT_EQUAL( 0, (int)myTestCPU->cc.bit.n );
    CPPUNIT_ASSERT_EQUAL( 0, (int)myTestCPU->cc.bit.z );
    CPPUNIT_ASSERT_EQUAL( 0, (int)myTestCPU->cc.bit.v );
}

/**
 * Test the LDD - Load into D - instruction.
 */
void
InstructionsTest::testLDD()
{
    // Test INDEXED mode:   LDD   2,Y
    //
    // Set up a word to test at address 0x205
    myTestCPU->write_word((Word)0x202, (Word)0xb3ff);
    // Set register D to something
    myTestCPU->d = 0x105;
    // Set register Y to point to that location minus 5
    myTestCPU->y = 0x200;
    // Two bytes of instruction
    myTestCPU->write((Word)0xB00, (Byte)0xEC);
    myTestCPU->write((Word)0xB01, (Byte)0x22);
    myTestCPU->pc = 0xB00;
    myTestCPU->execute();
    CPPUNIT_ASSERT_EQUAL( 0x200, (int)myTestCPU->y );
    CPPUNIT_ASSERT_EQUAL( 0xb3ff, (int)myTestCPU->d );
    CPPUNIT_ASSERT_EQUAL( 0xB02, (int)myTestCPU->pc );
    CPPUNIT_ASSERT_EQUAL( 1, (int)myTestCPU->cc.bit.n );
    CPPUNIT_ASSERT_EQUAL( 0, (int)myTestCPU->cc.bit.z );
    CPPUNIT_ASSERT_EQUAL( 0, (int)myTestCPU->cc.bit.v );

    // Test INDEXED mode:   LDD   -2,Y
    //
    myTestCPU->cc.all = 0;
    // Set up a word to test at address 0x205
    myTestCPU->write_word((Word)0x1FE, (Word)0x33ff);
    // Set register Y to point to that location minus 5
    myTestCPU->y = 0x200;
    // Two bytes of instruction
    myTestCPU->write((Word)0xB00, (Byte)0xEC);
    myTestCPU->write((Word)0xB01, (Byte)0x3E);
    myTestCPU->pc = 0xB00;
    myTestCPU->execute();
    CPPUNIT_ASSERT_EQUAL( 0x200, (int)myTestCPU->y );
    CPPUNIT_ASSERT_EQUAL( 0x33ff, (int)myTestCPU->d );
    CPPUNIT_ASSERT_EQUAL( 0xB02, (int)myTestCPU->pc );
    CPPUNIT_ASSERT_EQUAL( 0, (int)myTestCPU->cc.bit.n );
    CPPUNIT_ASSERT_EQUAL( 0, (int)myTestCPU->cc.bit.z );
    CPPUNIT_ASSERT_EQUAL( 0, (int)myTestCPU->cc.bit.v );

    // Test INDEXED mode:   LDD   ,--Y (decrement Y by 2 before loading D)
    //
    myTestCPU->cc.all = 0;
    // Set up a word to test at address 0x205
    myTestCPU->write_word((Word)0x200, (Word)0x31ff);
    // Set register Y to point to that location minus 5
    myTestCPU->y = 0x202;
    // Two bytes of instruction
    myTestCPU->write((Word)0xB00, (Byte)0xEC);
    myTestCPU->write((Word)0xB01, (Byte)0xA3);
    myTestCPU->pc = 0xB00;
    myTestCPU->execute();
    CPPUNIT_ASSERT_EQUAL( 0x200, (int)myTestCPU->y );
    CPPUNIT_ASSERT_EQUAL( 0x31ff, (int)myTestCPU->d );
    CPPUNIT_ASSERT_EQUAL( 0xB02, (int)myTestCPU->pc );
    CPPUNIT_ASSERT_EQUAL( 0, (int)myTestCPU->cc.bit.n );
    CPPUNIT_ASSERT_EQUAL( 0, (int)myTestCPU->cc.bit.z );
    CPPUNIT_ASSERT_EQUAL( 0, (int)myTestCPU->cc.bit.v );
}

void
InstructionsTest::testLEAX_PCR()
{
    Byte instructions[] = {
        0x30, // LEAX        >$0013,PCR
        0x8D,
        0xFE,
        0x49
    };

    int offset = 0x10000 - 0xfe49;
    // Negate 0
    loadProg(instructions, sizeof instructions);
    myTestCPU->execute();
    CPPUNIT_ASSERT_EQUAL( LOCATION + 4 - offset, (int)myTestCPU->x );
    CPPUNIT_ASSERT_EQUAL( LOCATION + 4, (int)myTestCPU->pc );

    // LEAX        <$93A,PCR
    myTestCPU->write((Word)0x0846, (Byte)0x30);
    myTestCPU->write((Word)0x0847, (Byte)0x8C);
    myTestCPU->write((Word)0x0848, (Byte)0xF1);
    myTestCPU->pc = 0x0846;
    myTestCPU->execute();
    CPPUNIT_ASSERT_EQUAL( 0x0849, (int)myTestCPU->pc );
    CPPUNIT_ASSERT_EQUAL( 0x083a, (int)myTestCPU->x );
}

/**
 * Test the LSL - Logical Shift Left instruction.
 */
void
InstructionsTest::testLSL()
{
    // Logical Shift Left of 0xff
    myTestCPU->cc.all = 0;
    myTestCPU->a = 0xff;
    myTestCPU->lsla();
    CPPUNIT_ASSERT_EQUAL( 0xfe, (int)myTestCPU->a );
    CPPUNIT_ASSERT_EQUAL( 0x09, (int)myTestCPU->cc.all );
//  CPPUNIT_ASSERT_EQUAL( 0, (int)myTestCPU->cc.bit.v );
//  CPPUNIT_ASSERT_EQUAL( 0, (int)myTestCPU->cc.bit.n );

    // Logical Shift Left of 1
    myTestCPU->cc.bit.c = 0;
    myTestCPU->cc.bit.v = 1;
    myTestCPU->a = 1;
    myTestCPU->lsla();
    CPPUNIT_ASSERT_EQUAL( 0x02, (int)myTestCPU->a );
    CPPUNIT_ASSERT_EQUAL( 0, (int)myTestCPU->cc.all );
//  CPPUNIT_ASSERT_EQUAL( 0, (int)myTestCPU->cc.bit.c );
//  CPPUNIT_ASSERT_EQUAL( 0, (int)myTestCPU->cc.bit.v );

    // Logical Shift Left of 0xB8
    myTestCPU->cc.bit.c = 0;
    myTestCPU->cc.bit.v = 0;
    myTestCPU->a = 0xB8;
    myTestCPU->lsla();
    CPPUNIT_ASSERT_EQUAL( 0x70, (int)myTestCPU->a );
    CPPUNIT_ASSERT_EQUAL( 0x03, (int)myTestCPU->cc.all );
//  CPPUNIT_ASSERT_EQUAL( 1, (int)myTestCPU->cc.bit.c );
//  CPPUNIT_ASSERT_EQUAL( 1, (int)myTestCPU->cc.bit.v );
}

/**
 * Test the NEG - Negate instruction.
 */
void
InstructionsTest::testNEG()
{
    // Negate 0
    myTestCPU->a = 0;
    myTestCPU->nega();
    CPPUNIT_ASSERT_EQUAL( 0, (int)myTestCPU->a );
    CPPUNIT_ASSERT_EQUAL( 0, (int)myTestCPU->cc.bit.c );

    // Negate 1
    myTestCPU->a = 1;
    myTestCPU->nega();
    CPPUNIT_ASSERT_EQUAL( 0xFF, (int)myTestCPU->a );
    CPPUNIT_ASSERT_EQUAL( 1, (int)myTestCPU->cc.bit.c );

    // Negate 2
    myTestCPU->a = 2;
    myTestCPU->nega();
    CPPUNIT_ASSERT_EQUAL( 0xFE, (int)myTestCPU->a );
    CPPUNIT_ASSERT_EQUAL( 1, (int)myTestCPU->cc.bit.c );
}

/**
 * Test the PULS - Pull registers from Hardware Stack - instruction.
 * The stack grows downwards, and this means that when you PULL, the
 * stack pointer is increased.
 */
void
InstructionsTest::testPULS()
{
    // Test: PULS PC,Y
    //
    // Set up stored Y value at 0x205
    myTestCPU->write_word((Word)0x205, (Word)0xb140);
    // Set up stored PC value at 0x207
    myTestCPU->write_word((Word)0x207, (Word)0x04ff);
    // Set Y to something benign
    myTestCPU->y = 0x1115;
    // Set register S to point to 0x205
    myTestCPU->s = 0x205;
    myTestCPU->cc.all = 0x0f;
    // Two bytes of instruction
    myTestCPU->write((Word)0xB00, (Byte)0x35); // PUL
    myTestCPU->write((Word)0xB01, (Byte)0xA0); // PC,Y
    myTestCPU->pc = 0xB00;
    myTestCPU->execute();
    CPPUNIT_ASSERT_EQUAL( 0xb140, (int)myTestCPU->y );
    CPPUNIT_ASSERT_EQUAL( 0x04ff, (int)myTestCPU->pc );
    CPPUNIT_ASSERT_EQUAL( 0x0f, (int)myTestCPU->cc.all );

}

/**
 * Test the instruction TST.
 * TST: The Z and N bits are affected according to the value
 * of the specified operand. The V bit is cleared.
 */
void
InstructionsTest::testTST()
{
    // Test a = 0xff
    myTestCPU->cc.all = 0;
    myTestCPU->a = 0xff;
    myTestCPU->tsta();
    CPPUNIT_ASSERT_EQUAL( 0xff, (int)myTestCPU->a );
    CPPUNIT_ASSERT_EQUAL( 1, (int)myTestCPU->cc.bit.n );
    CPPUNIT_ASSERT_EQUAL( 0, (int)myTestCPU->cc.bit.z );
    CPPUNIT_ASSERT_EQUAL( 0, (int)myTestCPU->cc.bit.v );

    //  Test a = 0x01 and V set
    myTestCPU->cc.all = 0;
    myTestCPU->cc.bit.v = 1;
    myTestCPU->a = 0x01;
    myTestCPU->tsta();
    CPPUNIT_ASSERT_EQUAL( 0x01, (int)myTestCPU->a );
    CPPUNIT_ASSERT_EQUAL( 0, (int)myTestCPU->cc.all );
//  CPPUNIT_ASSERT_EQUAL( 0, (int)myTestCPU->cc.bit.c );
//  CPPUNIT_ASSERT_EQUAL( 0, (int)myTestCPU->cc.bit.v );

    // Test a = 0x00
    myTestCPU->cc.all = 0;
    myTestCPU->a = 0x00;
    myTestCPU->tsta();
    CPPUNIT_ASSERT_EQUAL( 0, (int)myTestCPU->cc.bit.n );
    CPPUNIT_ASSERT_EQUAL( 1, (int)myTestCPU->cc.bit.z );
    CPPUNIT_ASSERT_EQUAL( 0, (int)myTestCPU->cc.bit.v );
    
    // Test indirect mode: TST ,Y
    // Set up a byte to test at address 0x205
    myTestCPU->write((Word)0x205, (Byte)0xff);
    // Set register Y to point to that location
    myTestCPU->y = 0x205;
    // Two bytes of instruction
    myTestCPU->write((Word)0xB00, (Byte)0x6d);
    myTestCPU->write((Word)0xB01, (Byte)0xa4);
    myTestCPU->pc = 0xB00;
    myTestCPU->execute();
    CPPUNIT_ASSERT_EQUAL( 1, (int)myTestCPU->cc.bit.n );
    CPPUNIT_ASSERT_EQUAL( 0, (int)myTestCPU->cc.bit.z );
    CPPUNIT_ASSERT_EQUAL( 0, (int)myTestCPU->cc.bit.v );

}

