/*
    (c) 2001-2005 Soren Roug

    This file is part of Osnine.

    Osnine is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    Osnine is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with Osnine; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

    $Id: os9io.cpp 60 2011-12-04 21:44:59Z roug $
*/

#include <cstdlib>
#include <cstdio>
#include <cstring>
using namespace std;
#include <ctype.h>
#include <time.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>
#include "mc6809.h"
#include "devdrvr.h"
#include "os9krnl.h"
#include "errcodes.h"


//------------------------ I/O calls -----------------------

/**
 * I$DUP - Duplicate a path number.
 *
 * After a cursory inspection of the disassembled shell and having some
 * trouble with basic09, I've come to the conclusion that System Manager's
 * Manual is wrong. The new path number is returned in register a.
 */
void os9::i_dup()
{
    Byte t;

    if(debug_syscall)
        fprintf(stderr,"os9::i_dup: %d ", a);

    for(t = 0; t < NumPaths; t++)
        if(paths[t] == NULL) {
            paths[t] = paths[a];
            paths[a]->usecount++;
            break;
        }
    if(t == NumPaths) {
        sys_error(200);
        return;
    }
    a = t;
    if(debug_syscall)
        fprintf(stderr,"=> %d\n", a);
}

/**
 * I$GETSTT - Get status of path number in register A.
 */
void os9::i_getstt()
{
    if(debug_syscall)
        fprintf(stderr,"os9::i_getstt: FD=%d opcode %d\n", a, b);

    paths[a]->errorcode = 0;
    paths[a]->getstatus(this);
}

/**
 * Setstatus.
 */
void os9::i_setstt()
{
    if(debug_syscall)
        fprintf(stderr,"os9::i_setstt: FD=%d opcode %d\n", a, b);
    paths[a]->errorcode = 0;
    paths[a]->setstatus(this);
}

/**
 * Make directory.
 */
void os9::i_mdir()
{
    Byte upath[512];
    devdrvr *dev;

    x += getpath(&memory[x], upath, 0);

    if(debug_syscall)
        fprintf(stderr,"os9::i_mdir: %s\n",(char*)upath);

    dev = find_device(upath);
    if(!dev) {
        sys_error(E_MNF);
        return;
    }
    sys_error(dev->makdir((char*)&upath[strlen(dev->mntpoint)], 0777));
    /* FIXME: mode bits */
}

/**
 * I$DELETEX.
 */
void os9::i_deletex(int xdir)
{
    Byte upath[512];
    devdrvr *dev;

    x += getpath(&memory[x], upath,(xdir)?(a&4):0);

    if(debug_syscall)
        fprintf(stderr,"os9::i_deletex: %s\n",(char*)upath);

    dev = find_device(upath);
    if(!dev) {
        sys_error(E_MNF);
        return;
    }
    sys_error(dev->delfile((char*)&upath[strlen(dev->mntpoint)]));
}

/**
 * I$OPEN - Open a file.
 * input  (X) = Address of pathlist
 * input  (A) = Access mode (D S PE PW PR E W R)
 * output (X) = Updated past pathlist (trailing spaces skipped)
 * outpu  (A) = Path number
 */
void os9::i_open(int create)
{
    Byte upath[512];
    devdrvr *dev;
    int mode = a;

    x += getpath(&memory[x], upath,(a & 4));

    if(debug_syscall)
        fprintf(stderr,"os9::i_open: %s (%s) mode %03o",
           (char*)upath, create?"create":"open", mode);

    dev = find_device(upath);
    if(!dev) {
        sys_error(E_MNF);
        return;
    }

    for(a = 0; a < NumPaths; a++) {
        if(paths[a] == NULL) {
            paths[a] =dev->open((char*)&upath[strlen(dev->mntpoint)]
                  , mode, create);

            if(paths[a] == NULL) {
                sys_error(E_PNNF);
                return;
            }
            break;
        }
    }
    if(a == NumPaths)
        sys_error(200);

    if(debug_syscall)
        fprintf(stderr,"= %d\n", a);
}

/**
 * Read a line.
 */
void os9::i_rdln()
{
    int c;

    if(debug_syscall > 1)
        fprintf(stderr,"os9::i_rdln: FD=%d pos=%x len=%d ", a, x , y);

    c = paths[a]->readln(&memory[x], y);
    if(c == -1) {
        sys_error(paths[a]->errorcode);
        if(debug_syscall > 1)
            fprintf(stderr,"error = %d\n", b);
        return;
    }
    y = (Word)c;
    if(debug_syscall > 1)
        fprintf(stderr,"ret = %d\n", y);
}

/**
 * Read some bytes.
 */
void os9::i_read()
{
    int c;

    if(debug_syscall > 1)
        fprintf(stderr,"os9::i_read: FD=%d pos=0x%x len=#%d ", a, x , y);

    c = paths[a]->read(&memory[x], y);
    if(c == -1) {
        sys_error(paths[a]->errorcode);
        if(debug_syscall > 1)
            fprintf(stderr,"error = %d\n", b);
        return;
    }
    y = (Word)c;
    if(debug_syscall > 1)
        fprintf(stderr,"ret = %d\n", y);
}

/**
 * I$SEEK: Seek in a random access file.
 */
void os9::i_seek()
{
    if(debug_syscall)
        fprintf(stderr,"os9::i_seek: FD=%d pos=%04X%04X\n", a, x, u);
    paths[a]->seek((x << 16) + u);
}

/**
 * I$WRLN - Write a line of text. The file descriptor is in register A.
 */
void os9::i_wrln()
{
    if(debug_syscall > 1)
        fprintf(stderr,"os9::i_wrln: FD=%d y=%d x=%04x %c%c%c...\n", a, y, x,
          read(x), read(x+1), read(x+2));

    paths[a]->errorcode = 0;
    y = paths[a]->writeln(&memory[x], y); // Return number of bytes written
    sys_error(paths[a]->errorcode);
}

/**
 * I$WRITE - Write some text. The file descriptor is in register A.
 */
void os9::i_write()
{
    if(debug_syscall > 1)
        fprintf(stderr,"os9::i_write: FD=%d y=%d x=%04x %c%c%c...\n", a, y, x,
          read(x), read(x+1), read(x+2));

    paths[a]->errorcode = 0;
    y = paths[a]->write(&memory[x], y); // Return number of bytes written
    sys_error(paths[a]->errorcode);
}

/**
 * Close a file descriptor.
 */
void os9::i_close()
{
    if(debug_syscall)
        fprintf(stderr,"os9::i_close: FD=%d\n", a);
    if(paths[a] == NULL) {
        sys_error(E_BPNum);
        return;
    }
    paths[a]->close();

    if(paths[a]->usecount == 0)
        delete paths[a];
    paths[a] = NULL;
}

/**
 * Change directory.
 * Contrary to what SYSMAN says, the output is that register x is updated past
 * the path.
 */
/*
 * FIXME: If the a&4 == 4 then set the exec dir bye changing the cxd
 * string.
 */
void os9::i_chgdir()
{
    Byte upath[512];
    Byte newcwd[256];
    devdrvr *dev;

    strcpy((char*)newcwd, cwd);
    getpath(&memory[x], upath,(a & 4));
    dev = find_device(newcwd);
    if(!dev) {
        sys_error(E_MNF);
        return;
    }
    if(sys_error(dev->chdir((char*)newcwd)) == 0)
        strcpy(cwd,(const char*)upath);
    fprintf(stderr,"Changing dir to %s\n", upath);
}
