/*
    (c) 2001-2005 Soren Roug

    This file is part of Osnine.

    Osnine is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    Osnine is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with Osnine; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

    $Id: os9krnl.cpp 60 2011-12-04 21:44:59Z roug $
*/

/**
 * @mainpage An OS9 level I emulator that runs under UNIX/Linux
 * <h2>Ambition level</h2>
 * To be able to run OS9 level I programs without the OS9 kernel. More
 * specifically: to be able to run BASIC09, C-compiler, Pascal-compiler and
 * the assembler so it is possible to write programs that is executed on a
 * "raw" 6809 emulator with the full os9 kernel installed. This implies
 * that a lot of other simple programs will also work, but not all.
 *
 */
#include <cstdio>
#include <cstdlib>
#include <cstring>
using namespace std;
#include <signal.h>
#include <ctype.h>
#include <time.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>
#include <dirent.h>
#include <errno.h>
#include <sys/wait.h>
#include "mc6809.h"
#include "devdrvr.h"
#include "devunix.h"
#include "os9krnl.h"
#include "errcodes.h"

/**
 * Print an OS9 string.
 * An os9 string is terminated with highorder bit set
 * This helpful function prints it out.
 */

/*
static void
print_os9string(FILE *out,Byte *str)
{
    while (*str < 128)
        putc(*str++, out);
    putc(*str & 127, out);
}
*/

/**
 * Compare two os9 caseinsensitive strings. A string is terminated with
 * highorder bit set, CR or NULL
 */
int os9::os9strcmp(Word s,Word t)
{
    for(;(memory[s]&95) == (memory[t]&95); s++, t++)
        if (memory[s] > 127 || memory[s] < 32)
            return 0;
    return (memory[s]&95) - (memory[t]&95);
}

/**
 * Find the device driver that handles a file with that pathname.
 */
devdrvr *os9::find_device(Byte *path)
{
    int i;

    for(i = 0; i < max_devs; i++)
       if (devices[i] && strncasecmp(devices[i]->mntpoint, (char*)path,
                strlen(devices[i]->mntpoint)) == 0)
           return devices[i];
    if (debug_syscall)
        fprintf(stderr,"No driver for %s\n", path);
    return 0;
}

/**
 * Open the root directory in UNIX and mount all directories as
 * OS9 devices.
 */
void os9::mount_allunix(void)
{
    DIR *dirp;
    struct dirent *dp;
    struct stat statbuf;
    char buf[1024];

    dirp = opendir("/");
    while ((dp = readdir(dirp))) {
        if (*dp->d_name == '.')   // Don't do hidden files
            continue;
        sprintf(buf,"/%s", dp->d_name);
        if (stat(buf, &statbuf) == -1)
            continue;
        if (S_ISDIR(statbuf.st_mode)) {
            if ( dev_end < max_devs )
                devices[dev_end++] = new devunix(buf, buf);
            else
                fprintf( stderr, "Overflow: %s device not added as OS9 device\n", buf );
        }
    }
    closedir(dirp);
    return ;
}

/**
 * Load RC file.
 * This function should read a configuration file in the user's home
 * directory. It could be called directly from the constructor.
 * The idea is to specify where /d0, /h0 is in the UNIX hierarchy.
 */
void os9::loadrcfile(void)
{
// Build a list of devices that map to a point in the UNIX filesystem
    char *homedir;

    if ((char *)getenv("OSNINEDIR") != NULL) {
        homedir = (char*)malloc(strlen(getenv("OSNINEDIR")) + 1);
        sprintf(homedir, "%s", getenv("OSNINEDIR"));
    } else {
        homedir = (char *)malloc(strlen(getenv("HOME")) + 5);
        sprintf(homedir, "%s/OS9", getenv("HOME"));
    }

    dev_end = 0;
    // Create a device called /term to match the UNIX controlling tty
    devices[dev_end++] = new devterm("/term","/dev/tty");
    // Create pseudo disks offset at $HOME/OS9
    // We create for all commonly used disks.
    devices[dev_end++] = new devunix("/dd", homedir); // Default drive
    devices[dev_end++] = new devunix("/h0", homedir);
    devices[dev_end++] = new devunix("/d0", homedir);
    devices[dev_end++] = new devunix("/d1", homedir);
    // Create the pipe device
    devices[dev_end++] = new devpipe("/pipe","");
    // Create devices for all UNIX directories we might be interested in
    // This is so a UNIX path name is directly equivalent to an OS9 pathname.
    // Used when we set the current working directory
    mount_allunix();
    /*
    devices[dev_end++] = new devunix("/home", "/home");
    devices[dev_end++] = new devunix("/usr", "/usr");
    devices[dev_end++] = new devunix("/etc", "/etc");
    */

    // Set the current execution directory.
    strcpy(cxd,"/dd/CMDS");
}

#define STARTPROG 0x0800
#define TOPMEM    0xfa00
const int mdirstart = 0x300;
const int mdirend   = 0x400;

// Constructor
os9::os9()
{
    devterm *tmpdev = new devterm("/term","/dev/tty");

    debug_syscall = 0;
    for(int inx = 0; inx < NumPaths; inx++) {
        paths[inx] = NULL;
    }
    // Set up stdin, stdout and stderr.
    paths[0] = tmpdev->open(stdin);
    paths[1] = tmpdev->open(stdout);
    paths[2] = tmpdev->open(stderr);

    // Set the CWD to what we have in UNIX
    getcwd(cwd, 1024);

    // Load the configuration file
    loadrcfile();

    init_mm();
    init_proc();

    // Setting the memory allocation bitmap SYSMAN 3.3
    // Sets the first 8 pages
    x = bitmapstart;
    d = 0;
    y = 0x100;
    f_delbit();

    d = 0;
    y = 8;
    f_allbit();

//  d = STARTPROG >> 8;
//  y = (TOPMEM - STARTPROG) >> 8;
//  f_delbit();
    d = TOPMEM >> 8;
    y = 6;
    f_allbit();

    // Set the top memory for the application
    write_word(D_MLIM,TOPMEM);

    // Set location of module directory
    write_word(D_MDir, mdirstart);
    write_word(D_MDirE, mdirend);
}
/*
 * Destructor
 */
os9::~os9()
{
}

/**
 * Set debug level.
 */
void os9::setdebugcalls(int newval)
{
    debug_syscall = newval;
}

int os9::error_occurred()
{
    return cc.bit.c;
}

/**
 * Create a system error. Sets the carry bit and the code in the B register.
 */
int os9::sys_error(Byte errcode)
{
    if (errcode == 0)
        return 0;
    cc.bit.c = 1;
    b = errcode;
    return errcode;
}

/**
 * Get the size of the argument transferred from the shell.
 * We do not count the \r as part of the string.
 */
static int
parmsize(const char *s)
{
    int i = 0;
    while (*s++ != '\r')
        i++;
    return i;
}

/**
 * Copy the argument vector.
 * Not \0 terminated
 */
static void
parmcopy(Byte *to,Byte *from)
{
    while ((*to++ = *from++) != '\r')
        ;
}

/**
 * getpath: get the path into a UNIX form, take into account the
 * execution directory.
 * Caller must provide adequate space in pathname.
 * Return value is the end of the path. You usually set register x to that.
 */
Word os9::getpath(Byte *mem,Byte *pathname, int xdir)
{
    Byte *mp;

    /*
     * When you do a "load filename" in basic09, getpath gets
     * called with leading spaces in filename
     */
    for(mp=mem;*mp == ' '; mp++)
       ;

    // If the path is absolute, prepend the offset into the UNIX fs
    if (*mp == '/')
    {
        *pathname = '\0';
    }
    else
    {
        if (xdir)
            sprintf((char*)pathname,"%s/", cxd);
        else
            sprintf((char*)pathname,"%s/", cwd);
        pathname += strlen((const char*)pathname);
    }

    for(; *mp; mp++)
    {
        if (*mp <= '-' || *mp == '<' || *mp == '>')
            break;
        *pathname++ = *mp & 0x7f;
        if (*mp & 0x80)
            break;
    }
    *pathname++ = '\0';

    // Skip past spaces
    for(;*mp == ' '; mp++)
       ;
    return mp - mem;
}

/**
 * copytomemory: copy a UNIX string to memory at byte position
 * String is NULL-terminated, but becomes CR-terminated when
 * copied to OS9 memory.
 */
void os9::copytomemory(Word addr, Byte *from)
{
    Byte *to = &memory[addr];
    while ((*to = *from++) != '\0')
        to++;
    *to = '\r';
}

/**
 * Load a file into memory as a module.
 */
void os9::loadmodule(const char *filename, const char *parm)
{
    fdes        *fd;
    Word        addr;
    devdrvr *dev;
    int         val, i;
    Byte tmpfn[1024];

    getpath((Byte*)filename, tmpfn, 1);
    dev = find_device(tmpfn);
    if (!dev) {
        sys_error(E_MNF);
        return;
    }
    filename = (char*)&tmpfn[strlen(dev->mntpoint)];
    fd = dev->open(filename, 1, 0);
    if (!fd) {
        b = E_PNNF;
        f_perr();
        exit(EXIT_FAILURE);
    }

    /* Get memory for the program area */
    addr = STARTPROG;
    while ((val = fd->read(&memory[addr], 256)) >0 ) {
        addr+= val;
    }
    fd->close();
    if (fd->usecount == 0) delete fd;

    add_to_mdir(STARTPROG);     // Add module to moddir
    memmark(STARTPROG, addr);

    pc = STARTPROG + read_word(STARTPROG + 0x09);
    /* Get memory for the data area */
    // 0x0b is first byte of permanent storage size
    // 0x02 is first byte of module size
    y = uppermem = STARTPROG + ((read(STARTPROG + 0x0b)+1) << 8)
                 + ((read(STARTPROG + 0x02)+1) << 8);
/*
 * In case you want to give all memory to the application
 * uncomment the next line
 */
//y = uppermem = TOPMEM;

// Load the argument vector and set registers
// parm is already terminated with \r
    d = parmsize(parm) + 1;
    s = y - d;
    for(i = s; i < y ; i++)
        write(i, *parm++);
    u = lowermem = STARTPROG + ((read(STARTPROG + 0x02)+1) << 8);
    if (debug_syscall) {
        fprintf(stderr, "module size: %04X\n",  read_word((STARTPROG + 0x02)));
        fprintf(stderr, "execution offset: %04X\n",  read_word((STARTPROG + 0x09)));
        fprintf(stderr, "permanent storage size: %04X\n",  read_word((STARTPROG + 0x0b)));
    }
    x = s;
    dp = u >> 8;
    cc.bit.f = 0;
    cc.bit.i = 0;
    if (debug_syscall)
        fprintf(stderr,"loadmodule: PC:%04X U:%04X DP:%02X X:%04X Y:%04X S:%04X\n",
         pc, u, dp, x, y, s);
    memmark(lowermem, uppermem); // Mark data section used
}

static const char *errmsg[] = {
#include "errmsg.i"
};


/**
 * F$ERR - Print error message. We have included the text strings so it
 * looks like you have used 'printerr'.
 */
void os9::f_perr(void)
{
    Byte buf[128];
    // According to sysman, a holds the path number to write to,
    // but the shell never sets a.
    sprintf((char*)buf,"ERROR #%d %s\r", b, errmsg[b]);
    paths[2]->writeln(buf, strlen((char*)buf));
}

/**
 * F$CHAIN - We will only support OS9 programs.
 * Because the parameter area can be overwritten
 * when we load a new program, we make a copy
 * outside of the emulator's memory.
 */
void os9::f_chain()
{
    Byte parm[256];

    parmcopy(parm,&memory[u]);
    if (debug_syscall)
    {
        Byte prog[256];
        parmcopy(prog,&memory[x]);
        fprintf(stderr,"os9::f_chain: %s %s\n",
         (char*)prog, (char*)parm);
    }
    loadmodule((char*)&memory[x], (char*)parm);
}

/**
 * F$EXIT - Exit running program.
 */
void os9::f_exit()
{
    if (debug_syscall)
        fprintf(stderr,"os9::f_exit\n");

    if (b != 0)
        fprintf(stderr,"Exit code %d\n", b);
    exit(b);
}

/**
 * F$FORK - Fork a new process.
 * We will only support OS9 programs.
 * According to The BASIC09 Reference manual revision F,
 * Chapter 9, CHAIN statement; the fork only passes path 0, 1 and 2
 * to the child program. FIXME: I this time ignore that "feature".
 *  - INPUT:
 *    - (X) = Address of module name or file name
 *    - (Y) = Parameter area size
 *    - (U) = Beginning address of the parameter area
 *    - (A) = Language/type code
 *    - (B) = Optional data area size (pages)
 *  - OUTPUT:
 *    - (X) = Updated past the name string
 *    - (A) = New process ID number
 */
void os9::f_fork()
{
    int pid;
    Byte upath[512];
    Word memoryneed = b * 256;
    Word parmarea = u;
    Word parmsize = y;

    if (debug_syscall)
        fprintf(stderr,"os9::f_fork X:%04X Y:%04X U:%04X B:%02X\n", x, y, u, b);

    Word maddr = find_in_mdir(x);
    if (maddr == 0) // Not found in module dir
        f_load();
    else {
        u = maddr;
        y = u + memory[u+0x09] * 256 + memory[u+0x0a];
        a = memory[u+6];
        b = memory[u+7];
    }

    if ((pid = fork()) == 0)
    { // child
        d = memoryneed;
        f_srqmem(); // Request memory for data area. Returned in U
        dp = u >> 8;
        pc = y;
        x = s = parmarea; // Bottom of parameter area
        y = parmarea + parmsize; // Top of parameter area
        cc.bit.f = 0;
        cc.bit.i = 0;
        if (debug_syscall)
            fprintf(stderr,"f_fork: PC:%04X U:%04X DP:%02X X:%04X Y:%04X S:%04X\n",
             pc, u, dp, x, y, s);
    }
    else
    { // parent
        x += getpath(&memory[x], upath, 1);
        a = 22; // Return child's process id. FIXME: Set to 2
    if (debug_syscall)
        fprintf(stderr,"os9::f_fork end X:%04X A:%02X *X:%d\n", x, a, memory[x]);
    }

}
/**
 * F$SLEEP - Sleep X hundreds of a second.
 */
void os9::f_sleep()
{
    if (debug_syscall)
        fprintf(stderr,"os9::f_sleep X:%04X\n", x);
    if (x == 1)
    {
       x = 0;
       return; // Same as giving up the timeslice.
    }
    if (x == 0)
        wait((int*)0);
    else
        sleep(x / 100);
}

/**
 * F$UNLINK - Tells OS-9 that the module is no longer needed by the calling
 * process. The module's link count is decremented, and the module is
 * destroyed and its memory deallocated when the link count equals zero.
 * The module will not be destroyed if in use by any other process(es)
 * because its link count will be non-zero.
 * INPUT:
 * (U) = Address of module header
 */
void os9::f_unlink()
{
    int mdirp, newcnt;

    if (debug_syscall)
        fprintf(stderr,"os9::f_unlink: u=%04X\n", u);

    for(mdirp = mdirstart; mdirp < mdirend; mdirp +=4) {
        if (read_word(mdirp) == u) {
            newcnt = read(mdirp + 2);
            write(mdirp+2, newcnt - 1);
            if (newcnt <= 1) {
                write_word(mdirp, 0); // removes entry
                d = memory[u+2]*256 + memory[u+3]; // Module size
                f_srtmem();
            }
            return;
        }
    }
    sys_error(E_MNF);
}

/**
 * F$LINK - This system call causes OS-9 to search the module directory for a
 * module having a name, language and type as given in the parameters.
 * If found, the address of the module's header is returned in U, and
 * the absolute address of the module's execution entry point is
 * returned in Y (as a convenience: this and other information can be
 * obtained from the module header). The module's link count' is
 * incremented whenever a LINK references its name, thus keeping track
 * of how many processes are using the module. If the module requested
 * has an attribute byte indicating it is not sharable (meaning it is
 * not reentrant) only one process may link to it at a time.
 *  - INPUT:
 *    - (X) = Address of module name string
 *    - (A) = Language / type (0 = any language / type)
 *  - OUTPUT:
 *    - (X) = Advanced past module name
 *    - (Y) = Module entry point address
 *    - (U) = Address of module header
 *    - (A) - Language / type
 *    - (B) = Attributes / revision level
 */
/*
 * FIXME: NOT FINISHED
 */
void os9::f_link()
{
    int mdirp, mcand, mname, maddr;

    if (debug_syscall)
        fprintf(stderr,"os9::f_link: x=%04X a=%02X\n", x, a);
    mcand = x;
/*    f_prsnam(); */
    for(mdirp = mdirstart; mdirp < mdirend; mdirp +=4) {
        maddr = read_word(mdirp);
        if (maddr == 0)
            continue;
        mname = maddr + memory[maddr+4] * 256 + memory[maddr+5];
        if ( os9strcmp(mname, x) == 0 ) {
            int newcnt = read(mdirp+2) + 1;
            write(mdirp+2, newcnt);
            u = maddr;
            x = y;
            y = u + memory[u+0x09] * 256 + memory[u+0x0a];
            a = memory[u+6];
            b = memory[u+7];
            return;
        }
    }
    sys_error(E_MNF);
}

/**
 * F$LOAD - Load module(s) from a file.
 *  - INPUT:
 *    - (X) = Address of pathlist (file name)
 *    - (A) = Language / type (0 = any language / type)
 *  - OUTPUT:
 *    - (X) = Advanced past pathlist
 *    - (Y) = Primary module entry point address
 *    - (U) = Address of module header
 *    - (A) - Language / type
 *    - (B) = Attributes / revision level
 *
 * Opens a file specified by the pathlist, reads one or more memory
 * modules from the file into memory, then closes the file. All modules
 * loaded are added to the system module directory with a use count of 0,
 * and the first module read is LINKed. The parameters returned are the same
 * as the F$LINK call and apply only to the first module loaded.
 *
 * In order to be loaded, the file must have the "execute"
 * permission and contain a module or modules that have a proper module
 * header. The file will be loaded from the working execution directory
 * unless a complete pathlist is given.
 *
 * If you load a file with two modules, both modules will be added to the
 * module directory with a use count of 0. Then the first module will be
 * LINKed giving it a use count of 1. Second time you load the same file,
 * the first module will get a link count of 2, while the second will still
 * have link count 0.
 */
void os9::f_load()
{
    Byte upath[512];
    unsigned char modhead[14];
    devdrvr *dev;
    fdes *fd;
    int first = 1;
    Word modname, modsize;
    Byte langtype;

    getpath(&memory[x], upath, 1);
    f_prsnam();

    if (debug_syscall)
        fprintf(stderr,"os9::f_load: %s\n", (char*)upath);

    dev = find_device(upath);
    if (!dev) {
        sys_error(E_MNF);
        return;
    }
    fd = dev->open((char*)&upath[strlen(dev->mntpoint)], 5, 0);
    if (!fd) {
        sys_error(E_PNNF);
        return;
    }

    while (1) {
        if (fd->read(modhead, 14) == -1)
            break;
        d = modsize = modhead[2]*256 + modhead[3]; // Module size
        f_srqmem();                      // Request memory of D size
        memcpy(&memory[u], modhead, 14);   // copy the header
        fd->read(&memory[u+14], modsize - 14); // Read the rest
        add_to_mdir(u);
        if (first)
        {
            first = 0;
            modname = u + modhead[4]*256 + modhead[5];
            langtype = modhead[6];
        }
    }
    fd->close();
    if (fd->usecount == 0) delete fd;

    x = modname;
    a = langtype;  // (A) - Language / type
    f_link();
}

/**
 * f_prsnam: Parse name.
 * Names can have one to 29 characters. They must begin with an upper- or lower-case
 * letter followed by any combination of the following character classes:
 * uppercase letters [A-Z], lowercase letters [a-z], decimal digits [0-9],
 * underscore (_) and period (.)
 * INPUT:
 * (X) = Address of the pathlist
 * OUTPUT:
 * (X) = Updated path the optional "/"
 * (Y) = Address of the last character of the name + 1
 * (B) = Length of the name
 *
 * Unkown behaviour: If a '!' is encountered
 */
void os9::f_prsnam()
{
    Byte *p;

    if (debug_syscall)
        fprintf(stderr,"os9::f_prsnam: X:%4X\n", x);

    if (read(x) == '/' || isalnum(read(x)) || read(x) == '_' || read(x) == '.')
    {
        p = &memory[x];

        while (*p == '/')   // Skip slash(es)
            p++;

        x = p - memory;

        while (*p == '_' || *p == '.' || isalnum(*p) )
            p++;
        y = p - memory;
        b = y - x;
        if (debug_syscall) {
            int i;
            for(i = 0;i < b; i++)
                fputc(read(x+i), stderr);
            fputc('\n', stderr);
        }
    }
    else // We are not pointing to a pathname
    {
        while (read(x) == ' ' || read(x) == '\t') {
            x++;
        }
        sys_error(E_BNam);
        if (debug_syscall)
            fprintf(stderr,"(whitespace)\n");
    }
}

#define CRC24_POLY 0x800063L

typedef long crc24;

/**
 * Compute CRC24 sum.
 */
static crc24
compute_crc(unsigned long crc, unsigned char *octets, int len)
{
    int i;

    while (len--) {
        crc ^= (*octets++) << 16;
        for (i = 0; i < 8; i++) {
            crc <<= 1;
            if (crc & 0x1000000)
                crc ^= CRC24_POLY;
        }
    }
    return crc & 0xffffffL;
}

/**
 * F:CRC: This service request calculates the CRC (cyclic redundancy count)
 * for use by compilers, assemblers, or other module generators. The CRC
 * is calculated starting at the source address over "byte count" bytes,
 * it is not necessary to cover an entire module in one call, since the
 * CRC may be "accumulated" over several calls. The CRC accumulator can
 * be any three byte memory location and must be initialized to $FFFFFF
 * before the first F$CRC call.
 */
void os9::f_crc()
{
    unsigned long tmpcrc;

    tmpcrc = (read(u) << 16) + (read(u+1) << 8) + read(u+2);

    if (debug_syscall)
      fprintf(stderr,"os9::f_crc: X=%04x Y=%04x DP=%02x\nU=%04x start=%lx\n",
             x, y, dp, u, tmpcrc);
    tmpcrc = compute_crc(tmpcrc,&memory[x], (int)y);
    write(u+0, (tmpcrc >> 16) & 0xff);
    write(u+1, (tmpcrc >> 8) & 0xff);
    write(u+2, tmpcrc & 0xff);
}

/**
 * Get userid.
 */
void os9::f_id()
{
    if (debug_syscall)
        fprintf(stderr,"os9::f_id\n");
    a = 1;
    y = getuid() & 0xffff;
}

/**
 * Get date and time.
 */
void os9::f_time()
{
    struct tm *local_time;
    time_t now;

    if (debug_syscall)
        fprintf(stderr,"os9::f_time\n");
    now = time(NULL);           // F$Time
    local_time = localtime(&now);
    write(x+0, (Byte)local_time->tm_year % 100); // Two char year.
    write(x+1, (Byte)local_time->tm_mon + 1);
    write(x+2, (Byte)local_time->tm_mday);
    write(x+3, (Byte)local_time->tm_hour);
    write(x+4, (Byte)local_time->tm_min);
    write(x+5, (Byte)local_time->tm_sec);
}

/**
 * f_wait: Wait for a process.
 */
void os9::f_wait()
{
    int ret = 0;
    pid_t pid;

    if (debug_syscall)
        fprintf(stderr,"%d-os9::f_wait\n", getpid());

    pid = wait(&ret);

    if (pid == -1) {
        perror("Wait UNIX error:");
        sys_error(E_NoChld);
    }
    else
    {
// FIXME: For some strange reason the os9 shell doesn't care what value
// has been returned in accumulator A in F$Fork. It always expects the
// value 1 in A when returned from F$Wait.
// Deeply mysterious
        a = 1;
        b = ret & 0xff;
        if (debug_syscall)
            fprintf(stderr,"wait normal return: A:%d B:%d CC:%d\n", a, b, cc.bit.c);
    }

}

/**
 * Software Interrupt 2 is used for system calls. Next byte after is the OPCODE.
 */
void os9::swi2(void)
{
        cc.bit.c = 0;
        switch(fetch())
        {
        case 0x00:
            f_link();
            break;
        case 0x01:
            f_load();
            break;
        case 0x02:
            f_unlink();
            break;
        case 0x03:
            f_fork();
            break;
        case 0x04:
            f_wait();
            break;

        case 0x05:
            f_chain();
            break;

        case 0x06:              // F$Exit
            f_exit();
            break;
        case 0x07:
            f_mem();
            break;
        case 0x09:
            if (debug_syscall)
                fprintf(stderr,"os9::Set intercept trap\n");
            break;
        case 0x0a:
            f_sleep();
            break;
        case 0x0c:
            f_id();
            break;
        case 0x0d:              // F$SPri
            /* Ignore */
            break;

        case 0x0f:              // F$Perr
            f_perr();
            break;
        case 0x10:              // F$Pnam
            f_prsnam();
            break;
        case 0x12:              // F$SchBit
            f_schbit();
            break;
        case 0x13:
            f_allbit();           // F$ABit
            break;
        case 0x14:
            f_delbit();           // F$DBit
            break;
        case 0x15:              // F$Time
            f_time();
            break;

        case 0x16:              // F$STim
            /* Ignore */
            break;

        case 0x17:
            f_crc();
            break;

        case 0x82:
            i_dup();
            break;

        case 0x83:
            i_open(1);          // I$Crea
            break;

        case 0x84:              // I$Open
            i_open(0);
            break;

        case 0x85:
            i_mdir();
            break;

        case 0x86:
            i_chgdir();
            break;

        case 0x87:
            i_deletex(0);
            break;

        case 0x88:
            i_seek();
            break;

        case 0x89:
            i_read();
            break;

        case 0x8a:
            i_write();
            break;

        case 0x8b:
            i_rdln();
            break;

        case 0x8c:
            i_wrln();
            break;

        case 0x8d:
            i_getstt();
            break;

        case 0x8e:
            i_setstt();
            break;

        case 0x8f:
            i_close();
            break;

        case 0x90:
            i_deletex(1);
            break;

        default:
            printf("Uncaught SWI2 call request %x\r\n", memory[--pc]);
            exit(0);
        }
}

/*
 * Module directory operations
 * The module directory consists of entries of two 2-byte values.
 * The first (16-bit) word is the address of the module in memory
 * The second word is the link count, but only the first byte is used!
 *   |ADDR|ADDR|LINK|----|
 *   |ADDR|ADDR|LINK|----|
 *   ...
 * If the ADDR is null, then the entry is to be ignored.
 */

/**
 * Find module in directory.
 */
Word os9::find_in_mdir(Word modname)
{
    int mdirp, mname, maddr;

    for(mdirp = mdirstart; mdirp < mdirend; mdirp +=4) {
        maddr = read_word(mdirp);
        if (maddr == 0)
            continue;
        mname = maddr + memory[maddr+4] * 256 + memory[maddr+5];
        if ( os9strcmp(mname, modname) == 0 ) {
            return maddr;
        }
    }
    return 0;
}

/**
 * Add module to directory.
 */
void os9::add_to_mdir(int modptr)
{
    int mdirp, found=0;

    for(mdirp=mdirstart; mdirp < mdirend; mdirp +=4) {
        if (read_word(mdirp) == modptr) {
            write(mdirp + 2, read(mdirp + 2) + 1);
            found = 1;
            break;
        }
    }
    /* Module didn't exist already, so create it.
     */
    if (found == 0)
        for(mdirp=mdirstart; mdirp < mdirend; mdirp +=4) {
            if (read_word(mdirp) == 0) {
                write_word(mdirp, modptr);
                write(mdirp + 2, 0);
                break;
            }
        }
}
