/*
    (c) 2001, 2002 Soren Roug

    This file is part of Osnine.

    Osnine is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    Osnine is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with Osnine; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

    $Id: os9mm.cpp 60 2011-12-04 21:44:59Z roug $
*/

#include <cstdlib>
#include <cstdio>
#include <cstring>
using namespace std;
#include <ctype.h>
#include <time.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>
#include "mc6809.h"
#include "devdrvr.h"
#include "os9krnl.h"
#include "errcodes.h"


/**
 * Memory housekeeping.
 * Maintain a bitmap of free memory at 0x200-0x21f incl.
 * This is mainly so extra modules/subroutes can be loaded.
 */
void os9::init_mm()
{
    // Set location of memory bitmap (0x200 - 0x220)
    write_word(D_FMBM,bitmapstart);
    write_word(D_FMBME,bitmapstart + 0x20);
}

/**
 * F$MEM: Used to expand or contract the process' data memory area. The new
 * size requested is rounded up to the next 256-byte page boundary.
 * Additional memory is allocated contiguously upward (towards higher
 * addresses), or deallocated downward from the old highest address. If
 * D = 0, then the current upper bound and size will be returned.
 * 
 * This request can never return all of a process' memory, or the
 * page in which its SP register points to.
 * 
 * In Level One systems, the request may return an error upon an
 * expansion request even though adequate free memory exists. This is
 * because the data area is always made contiguous, and memory requests
 * by other processes may fragment free memory into smaller, scattered
 * blocks that are not adjacent to the caller's present data area. Level
 * Two systems do not have this restriction because of the availability
 * of hardware for memory relocation, and because each process has its
 * own "address space".
 */
void os9::f_mem()
{
   if(d == 0)
   {
       y = uppermem;
       d = uppermem - lowermem;
   }
   else
   { // FIXME: check that the program requests less than what we have
       y = uppermem;
       d = uppermem - lowermem;
   }
}


/**
 * Mark pages of memory in bitmap. From and to are full addresses.
 */
void os9::memmark(int from, int to)
{
    from >>= 8;
    to >>= 8;

    for(; from <= to; from++) {
        memory[bitmapstart + (from/8)] |= (0x80 >> (from%8));
    }
}

/**
 * F$ALLBIT: This system mode service request sets bits in the allocation bit map
 * specified by the X register.  Bit numbers range from 0..N-1, where N is
 * the number of bits in the allocation bit map.
 * INPUT:
 * (X) = Base address of an allocation bit map.
 * (D) = Bit number of first bit to set.
 * (Y) = Bit count (number of bits to set).
 */
void os9::f_allbit()
{
    int from,to;

    from = d;
    to = d + y;
    while(from < to) {
        memory[x + (from/8)] |= (0x80 >> (from%8));
        from++;
    }
}

/**
 * F$DELBIT: This system mode service request is used to clear bits in the allocation
 * bit map pointed to by X.  Bit numbers range from 0..N-1, where N is the
 * number of bits in the allocation bit map.
 * INPUT:
 * (X) = Base address of an allocation bit map.
 * (D) = Bit number of first bit to clear.
 * (Y) = Bit count (number of bits to clear).
 */
void os9::f_delbit()
{
    int from, to;

    from = d;
    to = d + y;
    while(from < to) {
        memory[x + (from/8)] &= ~(0x80 >> (from%8));
        from++;
    }
}

/**
 * F$SchBit - Search bit map for a free area.
 * INPUT:
 * (X) = Beginning address of a bit map.
 * (D) = Beginning bit number.
 * (Y) = Bit count (free bit block size).
 * (U) = End of bit map address.
 * OUTPUT:
 * (D) = Beginning bit number.
 * (Y) = Bit count.
 * This system mode service request searches the specified allocation
 * bit map starting at the "beginning bit number" for a free
 * block (cleared bits) of the required length.
 *
 * If no block of the specified size exists, it returns with the
 * carry set, beginning bit number and size of the largest block.
 * FIXME: untested
 */
void os9::f_schbit()
{
    int from, to, bit;
    int len, candfrom, candlen;

    len = 0;
    from = d;
    to = d + y;
    while(from < to) {
        bit = memory[x + (from/8)] & (0x80 >> (from%8));
        if (bit) {
            /* bit is taken, we can go no further */
            if (len > candlen) {
                candlen = len;
                candfrom = from - len;
            }
        } else {
            /* bit is free */
            candlen++;
        }
        
        from++;
    }

    d = candfrom;
    y = candlen;
    if (candlen < y)
        sys_error(E_NoRam);
    return;
}



/**
 * F$SRQMEM - This system mode service request allocates a block of memory from
 * the top of available RAM of the specified size. The size requested is
 * rounded to the next 256 byte page boundary.
 *
 * find space in memory going from top to bottom
 *
 * Algoritm is first-fit
 * INPUT:
 * (D) = Byte count.
 * OUTPUT:
 * (U) = Beginning address of memory area.
 * (D) = Byte count rounded up to full page
 */
void os9::f_srqmem()
{
    int i,foundpages=0;
    int pages = (d+255) / 256;
    Word org_x, org_d, org_y;

    org_x = x;
    org_y = y;
    org_d = (d + 0xff) & 0xff00; // Round it up
    x = bitmapstart;
    for(i = 255; i >= 0; i--) {
        if (read(x + (i/8)) & (0x80 >> (i%8)))
            foundpages = 0;
        else
            foundpages++;
        if (foundpages == pages) {
            u = i * 256;
            d = i;
            y = pages;
            f_allbit();
            d = org_d;
            x = org_x;
            y = org_y;
            return;
        }
    }
    sys_error(E_NoRam);
    return;
}

/**
 * F$SRTMEM - This system mode service request is used to deallocate a block of contigous
 * 256 byte pages. The U register must point to an even page boundary.
 * INPUT:
 * (U) = Beginning address of memory to return.
 * (D) = Number of bytes to return.
 * OUTPUT:
 *  None.
 */
void os9::f_srtmem()
{
    Word org_x, org_d, org_y;

    org_x = x;
    org_y = y;
    org_d = d;
    x = bitmapstart;
    y = (d+255) / 256;
    d = u >> 8;
    f_delbit();
    x = org_x;
    y = org_y;
    d = org_d;
}

/**
 * F$ALL64 - Allocate 64 bytes for process/path descriptor
 * INPUT:
 * (X) = Base address of page table (zero if the page table has not yet been allocated).
 * OUTPUT:
 * (A) = Block number
 * (X) = Base address of page table
 * (Y) = Address of block.
 */
void os9::f_all64()
{
    Word org_u;
    Byte org_a;

    if (x != 0) {
        // Find an unused block
        for(a=1; 1; a++) {
            y = read(x + a / 4) << 8;
            if ( y == 0)  // No more blocks. Allocate one
                break;
            y += (a % 4) * 64;
            if ( read(y) == 0) { // Found an unused block
                return; 
            }
        }
    } else {
        a = 1;          // Block number 1
    }
    org_u = u;
    org_a = a;
    d = 1;          // Ask for one page
    f_srqmem();
    a = org_a;
    if ( x == 0)
        x = u;
    write(x + a / 4, u>>8); // Write MSB of x to page table;
    y = (read(x + a / 4) <<8) + (a % 4) * 64;
    u = org_u;
    return;
}

/**
 * F$RET64 - Deallocate a 64 byte memory block 
 * INPUT:
 * (X) = Address of the base page.
 * (A) = Block number.
 * FIXME: unfinished
 * I don't really know what to do here.
 */
void os9::f_ret64()
{
    y = (read(x + a / 4) << 8) + (a % 4) * 64;
    if ( y == 0)  // No more blocks. Error.
        return;
    write(y, 0);   // Clear the dirty flag.
}

