//
//
//	usim.cc
//
//	(C) R.P.Bellis 1994
//
//
//$Id: usim.cpp 56 2011-11-30 21:28:54Z roug $

#include <cstdlib>
#include <cstdio>
#include <cstring>
using namespace std;
#include "usim.h"

//----------------------------------------------------------------------------
// Generic processor run state routines
//----------------------------------------------------------------------------

/**
 * Run until illegal instrution is encountered.
 */
void USim::run(void)
{
	halted = 0;
	while (!halted) {
		execute();
	}
	status();
}

/**
 * Execute one instruction and then show status.
 */
void USim::step(void)
{
	execute();
	status();
}

/*
 * Set the halt flag.
 */
void USim::halt(void)
{
	halted = 1;
}

/**
 * Fetch one memory byte from program counter and increment program counter.
 */
Byte USim::fetch(void)
{
	Byte val = read(pc);
	pc += 1;

	return val;
}

/**
 * Fetch two memory bytes from program counter.
 */
Word USim::fetch_word(void)
{
	Word val = read_word(pc);
	pc += 2;

	return val;
}

/**
 * Invalid operation encountered. Halt the processor.
 */
void USim::invalid(const char *msg)
{
	fprintf(stderr, "\r\ninvalid %s : pc = [%04x], ir = [%04x]\r\n",
		msg ? msg : "",
		pc, ir);
	halt();
}

//----------------------------------------------------------------------------
// Primitive (byte) memory access routines
//----------------------------------------------------------------------------

/**
 * Single byte read from memory.
 */
Byte USim::read(Word offset)
{
	return memory[offset];
}

/**
 * Single byte write to memory.
 */
void USim::write(Word offset, Byte val)
{
	memory[offset] = val;
}

//----------------------------------------------------------------------------
// Processor loading routines
//----------------------------------------------------------------------------
static Byte fread_byte(FILE *fp)
{
	char			str[3];
	long			l;

	str[0] = fgetc(fp);
	str[1] = fgetc(fp);
	str[2] = '\0';

	l = strtol(str, NULL, 16);
	return (Byte)(l & 0xff);
}

static Word fread_word(FILE *fp)
{
	Word		ret;

	ret = fread_byte(fp);
	ret <<= 8;
	ret |= fread_byte(fp);

	return ret;
}

/**
 * Load file in Motorola S-record format into memory. The address
 * to load into is stored in the file.
 * Motorola's S-record format for output modules was devised for the
 * purpose of encoding programs or data files in a printable (ASCII)
 * format.  This allows viewing of the object file with standard
 * tools and easy file transfer from one computer to another, or
 * between a host and target.  An individual S-record is a single
 * line in a file composed of many S-records.
 *
 * S-Records are character strings made of several fields which
 * specify the record type, record length, memory address, data, and
 * checksum.  Each byte of binary data is encoded as a 2-character
 * hexadecimal number: the first ASCII character representing the
 * high-order 4 bits, and the second the low-order 4 bits of the
 * byte.
 */
void USim::load_srecord(const char *filename)
{
	FILE		*fp;
	int		done = 0;

	fp = fopen(filename, "r");
	if (!fp) {
		perror("filename");
		exit(EXIT_FAILURE);
	}

	while (!done) {
		Byte		n, t;
		Word		addr;
		Byte		b;

		while(fgetc(fp) != 'S' && !feof(fp)) // Look for the S
		      ;
		if(feof(fp))
		    return;
		t = fgetc(fp);			// Type
		n = fread_byte(fp);		// Length
		addr = fread_word(fp);		// Address
		n -= 2;
		if (t == '0') {
		        printf("Loading: ");
			while (--n) {
				b = fread_byte(fp);
				putchar(b);
			}
			putchar('\r');
			putchar('\n');
		} else if (t == '1') {
			while (--n) {
				b = fread_byte(fp);
				memory[addr++] = b;
			}
		
		} else if (t == '9') {
			pc = addr;
			done = 1;
		}
		// Read and discard checksum byte
		(void)fread_byte(fp);
	}
	fclose(fp);
}

/**
 * Load file in Intel Hex format into memory. The memory address
 * to load into is stored in the file.
 * There are several Intel hex formats available. The most common format used
 * is the 'Intel MCS-86 Hex Object'. This format uses the following
 * structure.
 * 
 * First char.     Start character
 * Next two char.  Byte count
 * next four char. Address
 * next two char.  Record type
 * last two char.  checksum
 */
void USim::load_intelhex(const char *filename)
{
	FILE		*fp;
	int		done = 0;

	fp = fopen(filename, "r");
	if (!fp) {
		perror("filename");
		exit(EXIT_FAILURE);
	}

	while (!done) {
		Byte		n, t;
		Word		addr;
		Byte		b;

		(void)fgetc(fp);
		n = fread_byte(fp);
		addr = fread_word(fp);
		t = fread_byte(fp);
		if (t == 0x00) {
			while (n--) {
				b = fread_byte(fp);
				memory[addr++] = b;
			}
		} else if (t == 0x01) {
			pc = addr;
			done = 1;
		}
		// Read and discard checksum byte
		(void)fread_byte(fp);
		if (fgetc(fp) == '\r') (void)fgetc(fp);
	}
	fclose(fp);
}

//----------------------------------------------------------------------------
// Word memory access routines for big-endian (Motorola type)
//----------------------------------------------------------------------------

/**
 * Read 16-bit word for big-endian (Motorola type).
 */
Word USimMotorola::read_word(Word offset)
{
	Word		tmp;

	tmp = read(offset++);
	tmp <<= 8;
	tmp |= read(offset);

	return tmp;
}

/**
 * Write 16-bit word for big-endian (Motorola type).
 */
void USimMotorola::write_word(Word offset, Word val)
{
	write(offset++, (Byte)(val >> 8));
	write(offset, (Byte)val);
}

//----------------------------------------------------------------------------
// Word memory access routines for little-endian (Intel type)
//----------------------------------------------------------------------------

/**
 * Read 16-bit word for little-endian (Intel type).
 */
Word USimIntel::read_word(Word offset)
{
	Word		tmp;

	tmp = read(offset++);
	tmp |= (read(offset) << 8);

	return tmp;
}

/**
 * Write 16-bit word for little-endian (Intel type).
 */
void USimIntel::write_word(Word offset, Word val)
{
	write(offset++, (Byte)val);
	write(offset, (Byte)(val >> 8));
}
